package test

import (
	"fmt"
	"github.com/stretchr/testify/require"

	"github.com/gruntwork-io/terratest/modules/http-helper"
	"github.com/gruntwork-io/terratest/modules/random"
	"github.com/gruntwork-io/terratest/modules/terraform"
	"testing"
	"time"
)

func TestAlbExample(t *testing.T) {

	t.Parallel()

	opts := &terraform.Options{
		// Powinieneś uaktualnić tę względną ścieżkę dostępu,
    // aby prowadziła do katalogu examples!
		TerraformDir: "../examples/alb",

		Vars: map[string]interface{}{
			"alb_name": fmt.Sprintf("test-%s", random.UniqueId()),
		},

	}

	// Usunięcie całej infrastruktury po zakończeniu testu.
	defer terraform.Destroy(t, opts)

	// Wdrożenie przykładu.
	terraform.InitAndApply(t, opts)

	// Pobranie adresu URL mechanizmu równoważenia obciążenia.
	albDnsName := terraform.OutputRequired(t, opts, "alb_dns_name")
	url := fmt.Sprintf("http://%s", albDnsName)

	// Sprawdzenie, czy domyślna akcja ALB działa prawidłowo i zwraca kod stanu 404.
	expectedStatus := 404
	expectedBody := "404: nie znaleziono strony"
	maxRetries := 10
	timeBetweenRetries := 10 * time.Second

	http_helper.HttpGetWithRetry(
		t,
		url,
		nil,
		expectedStatus,
		expectedBody,
		maxRetries,
		timeBetweenRetries,
	)

}

func TestAlbExamplePlan(t *testing.T) {
	t.Parallel()

	albName := fmt.Sprintf("test-%s", random.UniqueId())

	opts := &terraform.Options{
		// Powinieneś uaktualnić tę względną ścieżkę dostępu,
    // aby prowadziła do katalogu examples!
		TerraformDir: "../examples/alb",
		Vars: map[string]interface{}{
			"alb_name": albName,
		},
	}

	planString := terraform.InitAndPlan(t, opts)

	// Przykład pokazujący, jak sprawdzić liczniki add/change/destroy danych wyjściowych planu Terraform.
	resourceCounts := terraform.GetResourceCount(t, planString)
	require.Equal(t, 5, resourceCounts.Add)
	require.Equal(t, 0, resourceCounts.Change)
	require.Equal(t, 0, resourceCounts.Destroy)

	// Przykład sprawdzenia określonych wartości w danych wyjściowych terraform plan.
	planStruct :=
		terraform.InitAndPlanAndShowWithStructNoLogTempPlanFile(t, opts)

	alb, exists :=
		planStruct.ResourcePlannedValuesMap["module.alb.aws_lb.example"]
	require.True(t, exists, "aws_lb resource must exist")

	name, exists := alb.AttributeValues["name"]
	require.True(t, exists, "missing name parameter")
	require.Equal(t, albName, name)
}

